/*
 * tables2.c		- File system tables management for mke2fs
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation from mke2fs
 * 93/06/27	- Some cleanups for block sizes != 1024
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <linux/fs.h>
#include <linux/ext2_fs.h>

#include "bitops.h"
#include "e2fsprogs.h"
#include "mkfunc.h"
#include "mkvar.h"

/*
 * Note: this function works only if inodes_per_group is a multiple of 8
 *
 * This should not be a problem since the inodes count is rounded to fully
 * use the inode table blocks, a block is at least 1024 bytes and an inode
 * is 128 bytes
 */
static void write_desc (int dev)
{
	int i;
	char * block_bitmap = block_map;
	char * inode_bitmap = inode_map;
	char * inode_table = inode_buffer;
	char * full_bitmap = blkbuf;
	char * zeroed_inode_table = alloca (INODESPERGROUP * sizeof (struct ext2_inode));

	if (verbose)
		printf ("Writing bitmaps and tables\n");
	memset (full_bitmap, 0xff, block_size);
	memset (zeroed_inode_table, 0, INODESPERGROUP * sizeof (struct ext2_inode));
	if (INODESPERGROUP % 8)
		die ("inodes_per_group is not a multiple of 8", EXIT_ERROR);
	for (i = 0; i < group_desc_count; i++)
	{
		if (verbose)
			printf (" group %d: block bitmap at %d,",
				i, group_desc[i].bg_block_bitmap);
		if (lseek (dev, group_desc[i].bg_block_bitmap * block_size, SEEK_SET) !=
		    group_desc[i].bg_block_bitmap * block_size)
			die ("seek failed in write_desc", EXIT_ERROR);
		if (write (dev, full_bitmap, block_size) != block_size)
			die ("write of block bitmap failed in write_desc",
			     EXIT_ERROR);
		if (lseek (dev, group_desc[i].bg_block_bitmap * block_size, SEEK_SET) !=
		    group_desc[i].bg_block_bitmap * block_size)
			die ("seek failed in write_desc", EXIT_ERROR);
		if (write (dev, block_bitmap, BLOCKSPERGROUP / 8) !=
		    BLOCKSPERGROUP / 8)
			die ("write of block bitmap failed in write_desc",
			     EXIT_ERROR);
		block_bitmap += BLOCKSPERGROUP / 8;
		if (verbose)
			printf (" inode bitmap at %d,",
				group_desc[i].bg_inode_bitmap);
		if (lseek (dev, group_desc[i].bg_inode_bitmap * block_size, SEEK_SET) !=
		    group_desc[i].bg_inode_bitmap * block_size)
			die ("seek failed in write_desc", EXIT_ERROR);
		if (write (dev, full_bitmap, block_size) != block_size)
			die ("write of inode bitmap failed in write_desc",
			     EXIT_ERROR);
		if (lseek (dev, group_desc[i].bg_inode_bitmap * block_size, SEEK_SET) !=
		    group_desc[i].bg_inode_bitmap * block_size)
			die ("seek failed in write_desc", EXIT_ERROR);
		if (write (dev, inode_bitmap, INODESPERGROUP / 8) !=
		    INODESPERGROUP / 8)
			die ("write of inode bitmap failed in write_desc", EXIT_ERROR);
		inode_bitmap += INODESPERGROUP / 8;
		if (verbose)
			printf (" inode table at %d\n",
				group_desc[i].bg_inode_table);
		if (lseek (dev, group_desc[i].bg_inode_table * block_size, SEEK_SET) !=
		    group_desc[i].bg_inode_table * block_size)
			die ("seek failed in write_desc", EXIT_ERROR);
		if (write (dev, inode_table, inode_blocks_per_group * block_size) !=
		    inode_blocks_per_group * block_size)
			die ("write of inode table failed in write_desc", EXIT_ERROR);
		if (inode_table == inode_buffer)
			inode_table = zeroed_inode_table;
	}
}

void write_tables (int dev)
{
	int i;
	unsigned long block;

	/* Zeroes boot block */
	memset (blkbuf, 0, EXT2_MIN_BLOCK_SIZE);
	if (lseek (dev, 0, SEEK_SET) != 0)
		die ("seek 0 failed in write_tables", EXIT_ERROR);
	if (write (dev, blkbuf, EXT2_MIN_BLOCK_SIZE) != EXT2_MIN_BLOCK_SIZE)
		die ("write 0 failed in write_tables", EXIT_ERROR);
	/* Write master copy of the super block */
	if (lseek (dev, EXT2_MIN_BLOCK_SIZE, SEEK_SET) != EXT2_MIN_BLOCK_SIZE)
		die ("seek (first sb) failed in write_tables", EXIT_ERROR);
	if (write (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) !=
	    EXT2_MIN_BLOCK_SIZE)
		die ("unable to write super-block", EXIT_ERROR);
	/* Write copies of the super block */
	block = FIRSTBLOCK + BLOCKSPERGROUP;
	for (i = 1; i < group_desc_count; i++)
	{
		if (lseek (dev, block_size * block, SEEK_SET) !=
		    block_size * block)
			die ("seek (sb backup) failed in write_tables", EXIT_ERROR);
		if (write (dev, (char *) Super, EXT2_MIN_BLOCK_SIZE) !=
		    EXT2_MIN_BLOCK_SIZE)
			die ("unable to write super-block backup", EXIT_ERROR);
		block += BLOCKSPERGROUP;
	}
	write_desc (dev);
	block = FIRSTBLOCK + 1;
	for (i = 0; i < group_desc_count; i++)
	{
		if (lseek (dev, block_size * block, SEEK_SET) !=
		    block_size * block)
			die ("seek (desc) failed in write_tables", EXIT_ERROR);
		if (write (dev, (char *) group_desc, group_desc_size) !=
		    group_desc_size)
			die ("write of group descriptors failed\n", EXIT_ERROR);
		block += BLOCKSPERGROUP;
	}
}

/*
 * Create the group descriptors
 */

void make_group_desc (void)
{
	unsigned long i, j, k;
	unsigned long block;

	if (verbose)
		printf ("Making group descriptors\n");
	inode_blocks_per_group = (INODES / EXT2_INODES_PER_BLOCK (Super)) /
				 group_desc_count;
	for (i = 0, block = FIRSTBLOCK;
	     i < group_desc_count;
	     i++, block += BLOCKSPERGROUP)
	{
		for (j = 0; j < desc_blocks + 1; j++)
			mark_block (block + j);
		j = block + desc_blocks + 1;
		while (j < block + BLOCKSPERGROUP && block_in_use (j))
			j++;
		if (j >= block + BLOCKSPERGROUP)
		{
			printf ("group = %d\n", i);
			die ("Unable to find a block for the block bitmap", EXIT_ERROR);
		}
		group_desc[i].bg_block_bitmap = j;
		mark_block(j);
		if (verbose)
			printf (" group %d: block bitmap at %d,", i, j);
		while (j < block + BLOCKSPERGROUP && block_in_use (j))
			j++;
		if (j >= block + BLOCKSPERGROUP)
		{
			printf ("group = %d\n", i);
			die ("Unable to find a block for the inode bitmap", EXIT_ERROR);
		}
		group_desc[i].bg_inode_bitmap = j;
		mark_block(j);
		if (verbose)
			printf (" inode bitmap at %d,", j);
repeat:
		while (j < block + BLOCKSPERGROUP && block_in_use (j))
			j++;
		if (j >= block + BLOCKSPERGROUP)
		{
			printf ("group = %d\n", i);
			die ("Unable to find a block for the inode table", EXIT_ERROR);
		}
		for (k = 0; k < inode_blocks_per_group; k++)
			if (block_in_use (j + k))
			{
				j = j + k;
				goto repeat;
			}
		group_desc[i].bg_inode_table = j;
		for (k = 0; k < inode_blocks_per_group; k++)
			mark_block (j + k);
		if (verbose)
			printf (" inode table at %d (%d)\n",
				j, inode_blocks_per_group);
		group_desc[i].bg_free_blocks_count = 0;
		group_desc[i].bg_free_inodes_count = 0;
		group_desc[i].bg_used_dirs_count = 0;
	}
}

void count_free (void)
{
	int i, j;
	int count;
	int dirs_count;
	int block = FIRSTBLOCK;
	int ino = 1;
	int total_blocks_count = 0;
	int total_inodes_count = 0;
	int block_count;

	if (verbose)
		printf ("Computing free blocks and inodes count\n");
	for (i = 0; i < group_desc_count; i++)
	{
		count = 0;
		if (i == group_desc_count - 1)
			block_count = BLOCKS - block;
		else
			block_count = BLOCKSPERGROUP;
		for (j = 0; j < block_count; j++)
		{
			if (!block_in_use(block + j))
				count++;
		}
		group_desc[i].bg_free_blocks_count = count;
		if (verbose)
			printf (" group %d: %d free blocks,",
				i, group_desc[i].bg_free_blocks_count);
		total_blocks_count += count;
		block += BLOCKSPERGROUP;
		count = 0;
		dirs_count = 0;
		for (j = 0; j < INODESPERGROUP; j++)
			if (!inode_in_use(ino + j))
				count++;
		group_desc[i].bg_free_inodes_count = count;
		if (verbose)
			printf (" %d free inodes, %d directories\n",
				group_desc[i].bg_free_inodes_count,
				group_desc[i].bg_used_dirs_count);
		total_inodes_count += count;
		ino += INODESPERGROUP;
	}
	FREEBLOCKS = total_blocks_count;
	FREEINODES = total_inodes_count;
}

void setup_tables (void)
{
	int i;
	int log_size = 0;
	int size;
	int block_map_size;

	memset ((char *) Super, 0, EXT2_MIN_BLOCK_SIZE);
	MTIME = 0;
	WTIME = time (NULL);
	MAGIC = EXT2_SUPER_MAGIC;
	VALID = 1;

	size = block_size >> (EXT2_MIN_BLOCK_LOG_SIZE + 1);
	while (size)
	{
		log_size++;
		size >>= 1;
	}
	BLOCKSIZE = log_size;

	size = frag_size >> (EXT2_MIN_FRAG_LOG_SIZE + 1);
	log_size = 0;
	while (size)
	{
		log_size++;
		size >>= 1;
	}
	FRAGSIZE = log_size;

	BLOCKS = blocks / (block_size / EXT2_MIN_BLOCK_SIZE);
	RBLOCKS = (BLOCKS * reserved_ratio) / 100;
	FRAGSPERGROUP = blocks_per_group * (block_size / frag_size);
	BLOCKSPERGROUP = blocks_per_group;

	group_desc_count = (((BLOCKS - NORM_FIRSTBLOCK) * frags_per_block) /
			    BLOCKSPERGROUP);
	if (group_desc_count * BLOCKSPERGROUP < (BLOCKS - NORM_FIRSTBLOCK) *
	    frags_per_block)
		group_desc_count++;
	if (group_desc_count % EXT2_DESC_PER_BLOCK (Super))
		desc_blocks = (group_desc_count / EXT2_DESC_PER_BLOCK (Super)) + 1;
	else
		desc_blocks = group_desc_count / EXT2_DESC_PER_BLOCK (Super);
	group_desc_size = desc_blocks * block_size;

/* some magic nrs: 1 inode / 4096 bytes (now use the inode ratio) */
	INODES = (blocks * EXT2_MIN_BLOCK_SIZE) / inode_ratio;

/* Round the inodes count to fully use each group descriptor */
	if (INODES % group_desc_count)
		INODESPERGROUP = (INODES / group_desc_count) + 1;
	 else
		INODESPERGROUP = INODES / group_desc_count;

/* Round the inodes count to fully use each block in each descriptor */
	if (INODESPERGROUP % EXT2_INODES_PER_BLOCK (Super))
		INODESPERGROUP = ((INODESPERGROUP / EXT2_INODES_PER_BLOCK (Super)) + 1) *
				   EXT2_INODES_PER_BLOCK (Super);
	INODES = INODESPERGROUP * group_desc_count;

	FIRSTBLOCK = NORM_FIRSTBLOCK;

	inode_map = malloc (INODES / 8 + 1);
	if (!inode_map)
		die ("cannot allocate inode bitmap\n", EXIT_ERROR);
	memset (inode_map, 0xff, INODES / 8 + 1);
	for (i = EXT2_FIRST_INO; i <= INODES; i++)
		unmark_inode (i);

	block_map_size = (BLOCKS / 8 + 1);
	if (block_map_size % (block_size * 8))
		block_map_size = (block_map_size / (block_size * 8) + 1) *
				 (block_size * 8);
	block_map = malloc (block_map_size);
	if (!block_map)
		die ("cannot allocate block bitmap\n", EXIT_ERROR);
	memset (block_map, 0xff, block_map_size);

	bad_map = malloc (block_map_size);
	if (!bad_map)
		die ("cannot allocate bad block bitmap\n", EXIT_ERROR);
	memset (bad_map, 0, block_map_size);

	for (i = FIRSTBLOCK; i < BLOCKS; i++)
		unmark_block (i);

	inode_buffer = malloc (INODESPERGROUP * sizeof (struct ext2_inode));
	if (!inode_buffer)
		die ("Unable to allocate buffer for inodes", EXIT_ERROR);
	memset (inode_buffer, 0, INODESPERGROUP * sizeof (struct ext2_inode));

	group_desc = malloc (group_desc_size);
	if (!group_desc)
		die ("Unable to allocate buffer for groups descriptors", EXIT_ERROR);
	memset (group_desc, 0, group_desc_size);

	printf ("%d inodes\n", INODES);
	printf ("%d blocks\n", BLOCKS);
	printf ("%d blocks reserved for the super user\n", RBLOCKS);
	printf ("First data block=%d (%d)\n", FIRSTBLOCK, NORM_FIRSTBLOCK);
	printf ("Block size=%d (log=%d)\n",
		EXT2_MIN_BLOCK_SIZE << BLOCKSIZE, BLOCKSIZE);
	printf ("Fragment size=%d (log=%d)\n",
		EXT2_MIN_FRAG_SIZE << FRAGSIZE, FRAGSIZE);
	printf ("%d blocks group%s\n", group_desc_count, (group_desc_count > 1) ? "s" : "");
	printf ("%d blocks per group\n", BLOCKSPERGROUP);
	printf ("%d fragments per group\n", FRAGSPERGROUP);
	printf ("%d inodes per group\n", INODESPERGROUP);
}
