/*
 * inode.c		- Inode format dependent operations for e2fsck
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation from e2fsck
 */

#include <stdio.h>

#include <sys/stat.h>

#include <linux/fs.h>
#include <linux/ext2_fs.h>

#include "bitops.h"
#include "e2fsprogs.h"
#include "ckfunc.h"
#include "ckvar.h"

struct ext2_inode get_inode (int dev, unsigned long ino)
{
	struct ext2_inode inode;

	DEBUG(("DEBUG: get_inode (%d)\n", ino));
	if (!ino || ino > INODES)
	{
		inode.i_mode = 0;
		return inode;
	}
	total++;
	read_inode (dev, ino, &inode);
	if (!inode_count [ino])
	{
		if (!inode_in_use (ino))
		{
			printf ("Inode %d marked not used, but used for file '",
				ino);
			print_current_name ();
			printf ("'.\n");
			if (ask ("Mark in use", 1))
				mark_inode (ino);
		}
		if (S_ISDIR (inode.i_mode))
			directory++;
		else if (S_ISREG (inode.i_mode))
			regular++;
		else if (S_ISCHR (inode.i_mode))
			chardev++;
		else if (S_ISBLK (inode.i_mode))
			blockdev++;
		else if (S_ISLNK (inode.i_mode))
		{
			symlinks++;
			if (!inode.i_blocks)
				fast_symlinks++;
		}
		else if (S_ISFIFO (inode.i_mode))
			fifo++;
		else if (S_ISSOCK (inode.i_mode))
		        sockets++;
		else
		{
			print_current_name ();
			printf (" (inode %d) has bad mode (%o). ",
				ino, inode.i_mode);
			if (ask ("Set mode to file", 0))
			{
				inode.i_mode |= S_IFREG;
				write_inode (dev, ino, &inode);
			}
#if 0
			else
				return NULL;
#endif
		}
	}
	else
		links++;
	if (!++inode_count[ino])
	{
		printf ("Warning: inode count too big.\n");
		inode_count[ino]--;
	}
	return inode;
}

int add_block (int dev, unsigned long * znr, int badflg)
{
	int result;

	DEBUG(("DEBUG: add_block(&%d,%d)\n", *znr, badflg));
	result = check_block_nr (znr);
	if (!*znr || *znr >= BLOCKS)
		return result;
	if (!badflg && block_is_bad (*znr))
	{
		printf ("Bad block used in file `");
		print_current_name ();
		printf ("'. ");
		if (ask ("Clear", 1))
		{
			*znr = 0;
			changed = 1;
			return 1;
		}
	}
	else if (block_count[*znr])
	{
		printf ("Block %d has been used before. Now in file `", *znr);
		print_current_name ();
		printf ("'. ");
		if (ask ("Clear", 1))
		{
			*znr = 0;
			changed = 1;
			return 1;
		}
	}
	if (!block_in_use (*znr))
	{
		printf ("Block %d in file `", *znr);
		print_current_name ();
		printf ("' is marked not in use. ");
		if (ask ("Mark in use", 1))
			mark_block (*znr);
	}
	if (badflg)
	{
		DEBUG(("DEBUG: Marking %d as bad.\n", *znr));
		mark_bad (*znr);
	}
	if (!++block_count[*znr])
		block_count[*znr]--;
	return result;
}

int add_block_ind (int dev, unsigned long * znr, int badflg)
{
	char * blk = blkbuf + block_size;
	int i, result, chg_blk = 0;

	DEBUG(("DEBUG: add_block_ind (&%d,%d)\n", *znr, badflg));
	result = add_block (dev, znr, 0);
	if (!*znr || *znr >= BLOCKS || block_is_bad (*znr))
		return result;
	read_block (dev, znr, blk);
	for (i = 0; i < addr_per_block; i++)
		chg_blk |= add_block (dev, ((unsigned long *) blk) + i,
				      badflg);
	if (chg_blk)
		write_block (dev, *znr, blk);
	return result;
}

int add_block_dind (int dev, unsigned long * znr, int badflg)
{
	char * blk = blkbuf + block_size * 2;
	int i, result, blk_chg = 0;

	DEBUG(("DEBUG: add_block_dind (&%d,%d)\n", *znr, badflg));
	result = add_block (dev, znr, 0);
	if (!*znr || *znr >= BLOCKS || block_is_bad (*znr))
		return result;
	read_block (dev, znr, blk);
	for (i = 0; i < addr_per_block; i++)
		blk_chg |= add_block_ind (dev, ((unsigned long *) blk) + i,
					  badflg);
	if (blk_chg)
		write_block (dev, *znr, blk);
	return result;
}

int add_block_tind (int dev, unsigned long * znr, int badflg)
{
	char * blk = blkbuf + block_size * 3;
	int i, result, blk_chg = 0;

	DEBUG(("DEBUG: add_block_tind (&%d,%d)\n", *znr, badflg));
	result = add_block (dev, znr, 0);
	if (!*znr || *znr >= BLOCKS || block_is_bad (*znr))
		return result;
	read_block (dev, znr, blk);
	for (i = 0; i < addr_per_block; i++)
		blk_chg |= add_block_dind (dev, ((unsigned long *) blk) + i,
					   badflg);
	if (blk_chg)
		write_block (dev, *znr, blk);
	return result;
}

/*
 * mapped_read_block reads block nr blknr from the specified file.
 * it returns 1 if the inode has been changed due to bad block nrs
 */
int mapped_read_block (int dev, struct ext2_inode * inode,
		       unsigned long blknr, char * addr)
{
	unsigned long * ind  = (unsigned long *) (blkbuf + block_size);
	unsigned long * dind = (unsigned long *) (blkbuf + block_size * 2);
	unsigned long * tind = (unsigned long *) (blkbuf + block_size * 3);
	int result;

	DEBUG(("DEBUG: mapped_read_block(%d,%d,%d)\n", (int)inode,
		blknr, (int) addr));
	if (blknr < EXT2_NDIR_BLOCKS)
		return read_block (dev, inode->i_block + blknr, addr);
	blknr -= EXT2_NDIR_BLOCKS;
	if (blknr < addr_per_block)
	{
		result = read_block (dev, inode->i_block + EXT2_IND_BLOCK,
				     (char *) ind);
		if (read_block (dev, blknr + ind, addr))
		{
			result = 1;
			write_block (dev, inode->i_block[EXT2_IND_BLOCK],
				     (char *) ind);
		}
		return result;
	}
	blknr -= addr_per_block;
	if (blknr < addr_per_block * addr_per_block)
	{
		result = read_block (dev, inode->i_block + EXT2_DIND_BLOCK,
				     (char *) dind);
		if (read_block (dev, dind + blknr / addr_per_block, (char *) ind))
		{
			result = 1;
			write_block (dev, inode->i_block[EXT2_DIND_BLOCK],
				     (char *) dind);
		}
		if (read_block (dev, ind + blknr % addr_per_block, addr))
		{
			result = 1;
			write_block (dev, dind[blknr / addr_per_block],
				     (char *) ind);
		}
		return result;
	}
	blknr -= addr_per_block * addr_per_block;
	result = read_block (dev, inode->i_block + EXT2_TIND_BLOCK,
			     (char *) tind);
	if (read_block (dev, tind + (blknr >> 16), (char *) dind))
	{
		result = 1;
		write_block (dev, inode->i_block[EXT2_TIND_BLOCK], (char *) tind);
	}
	if (read_block (dev, dind + ((blknr >> 8) & (addr_per_block - 1)),
			(char *) ind))
	{
		result = 1;
		write_block (dev, tind[blknr >> 16], (char *) dind);
	}
	if (read_block (dev, ind + blknr % addr_per_block, addr))
	{
		result = 1;
		write_block (dev, dind[(blknr >> 8) & (addr_per_block - 1)],
			     (char *) ind);
	}
	return result;
}
