/*
 * cache.c		- Inode read/write operations for e2fsck
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation from e2fsck
 */

#include <unistd.h>

#include <linux/fs.h>
#include <linux/ext2_fs.h>

#include "e2fsprogs.h"
#include "ckfunc.h"
#include "ckvar.h"

/*
 * These functions manage a 1KB block cache used by the inode cache
 */
static unsigned long cache_block_nr = 0;
static unsigned char cache_block_dirt = 0;
static char cache_buffer[EXT2_MAX_BLOCK_SIZE];

static char * cache_read_block (int dev, unsigned long block_nr)
{
	if (block_nr != cache_block_nr)
	{
		if (cache_block_dirt)
		{
			if (lseek (dev, cache_block_nr * block_size, SEEK_SET) !=
				cache_block_nr * block_size)
				die ("seek failed in cache_read_block", EXIT_ERROR);
			if (write (dev, cache_buffer, block_size) != block_size)
				die ("write failed in cache_read_block", EXIT_ERROR);
		}
		cache_block_dirt = 0;
		cache_block_nr = block_nr;
		if (lseek (dev, cache_block_nr * block_size, SEEK_SET) !=
		    cache_block_nr * block_size)
			die ("seek failed in cache_read_block", EXIT_ERROR);
		if (read (dev, cache_buffer, block_size) != block_size)
			die ("read failed in cache_read_block", EXIT_ERROR);
	}
	return cache_buffer;
}

static void cache_write_block (int dev, unsigned long block_nr)
{
	if (block_nr != cache_block_nr)
		die ("cache_write_block: block_nr != cache_block_nr", EXIT_ERROR);
	cache_block_dirt = 1;
}

void cache_flush (int dev)
{
	if (cache_block_dirt)
	{
		if (lseek (dev, cache_block_nr * block_size, SEEK_SET) !=
		    cache_block_nr * block_size)
			die ("seek failed in cache_flush", EXIT_ERROR);
		if (write (dev, cache_buffer, block_size) != block_size)
			die ("write failed in cache_flush", EXIT_ERROR);
	}
	cache_block_dirt = 0;
}
		
/*
 * These functions manage a cache for the inodes
 */
void read_inode (int dev, unsigned long ino, struct ext2_inode * inode)
{
	unsigned long group;
	unsigned long block;
	unsigned long block_nr;
	int i;
	char * buffer;

	group = (ino - 1) / EXT2_INODES_PER_GROUP(Super);
	block = ((ino - 1) % EXT2_INODES_PER_GROUP(Super)) /
		EXT2_INODES_PER_BLOCK(Super);
	i = ((ino - 1) % EXT2_INODES_PER_GROUP(Super)) %
		EXT2_INODES_PER_BLOCK(Super);
	block_nr = group_desc[group].bg_inode_table + block;
	buffer = cache_read_block (dev, block_nr);
	memcpy (inode, (struct ext2_inode *) buffer + i,
		sizeof (struct ext2_inode));
}

void write_inode (int dev, unsigned long ino, struct ext2_inode * inode)
{
	unsigned long group;
	unsigned long block;
	unsigned long block_nr;
	int i;
	char * buffer;

	group = (ino - 1) / EXT2_INODES_PER_GROUP(Super);
	block = ((ino - 1) % EXT2_INODES_PER_GROUP(Super)) /
		EXT2_INODES_PER_BLOCK(Super);
	i = ((ino - 1) % EXT2_INODES_PER_GROUP(Super)) %
		EXT2_INODES_PER_BLOCK(Super);
	block_nr = group_desc[group].bg_inode_table + block;
	buffer = cache_read_block (dev, block_nr);
	memcpy ((struct ext2_inode *) buffer + i, inode,
		sizeof (struct ext2_inode));
	cache_write_block (dev, block_nr);
	changed = 1;
}
