/*
 * badblocks1.c		- Bad blocks management for e2fsck
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation from e2fsck
 */

#include <signal.h>
#include <stdio.h>
#include <unistd.h>

#include <linux/fs.h>
#include <linux/ext2_fs.h>

#include "bitops.h"
#include "e2fsprogs.h"
#include "ckfunc.h"
#include "ckvar.h"

/*
 * Perform a test of a block; return the number of blocks readable/writeable.
 */
static long do_test (int dev, char * buffer, int try,
		     unsigned long current_block)
{
	long got;

	DEBUG (("DEBUG: do_test (buf,%d,%d)\n", try, current_block));
	
	/* Seek to the correct loc. */
	if (lseek (dev, current_block * block_size, SEEK_SET) !=
	    current_block * block_size)
                 die ("seek failed during testing of blocks", EXIT_ERROR);

	/* Try the read */
	got = read (dev, buffer, try * block_size);
	if (got < 0)
		got = 0;	
	if (got & (block_size - 1))
		printf ("Weird values in do_test: probably bugs\n");
	got /= block_size;
	return got;
}

static unsigned long currently_testing = 0;

static void alarm_intr (int alnum)
{
	if (currently_testing >= BLOCKS)
		return;
	signal (SIGALRM, alarm_intr);
	alarm (5);
	if (!currently_testing)
		return;
	printf ("%d... ", currently_testing);
	fflush (stdout);
}

void test_blocks (int dev)
{
	int try;
	long got;
	int currently_bad = badblocks;

	DEBUG (("DEBUG: test_blocks ()\n"));
	if (verbose)
		printf ("Testing the disk for bad blocks\n");
	currently_testing = 0;
	if (verbose)
	{
		signal (SIGALRM, alarm_intr);
		alarm (5);
	}
	try = TEST_BUFFER_BLOCKS;
	while (currently_testing < BLOCKS)
	{
		if (currently_testing + try > BLOCKS)
			try = BLOCKS - currently_testing;
		if (currently_testing >= FIRSTBLOCK)
		{
			int i;

			/* Skip blocks that are known to be bad */
			while (currently_testing < BLOCKS &&
			       block_is_bad (currently_testing))
				currently_testing++;
			if (currently_testing >= BLOCKS)
				break;
			/* Shorten group if it contains a bad block */
			for (i = try - 1; i; i--)
			{
				if (block_is_bad (currently_testing + try - i))
				{
					try -= i;
					break;
				}
			}
		}
		else if (currently_testing + try > FIRSTBLOCK)
			try = FIRSTBLOCK - currently_testing;
		got = do_test (dev, blkbuf, try, currently_testing);
		currently_testing += got;
		if (got == try)
		{
			try = TEST_BUFFER_BLOCKS;
			continue;
		}
		else
			try = 1;
		DEBUG (("DEBUG: Only got %d\n", got));
		if (currently_testing < FIRSTBLOCK)
			die ("bad blocks before data-area: cannot fix!", EXIT_ERROR);
		mark_bad (currently_testing);
		/* If this block is in use, we'll warn about it later.
		   For now, just make sure it is marked as used. */
		if (!block_in_use (currently_testing))
		{
			mark_block (currently_testing);
			FREEBLOCKSCOUNT--;
		}
		currently_testing++;
	}
	if (verbose)
		printf ("\n");
	if (badblocks - currently_bad)
		printf ("found %d bad block%s\n", badblocks - currently_bad,
			(badblocks - currently_bad != 1) ? "s" : "");
}

void read_bad_blocks (int dev)
{
	struct ext2_inode inode;
	int i;

	DEBUG (("DEBUG: read_bad_blocks()\n"));
	if (verbose)
		printf ("Reading bad blocks list\n");
	read_inode (dev, EXT2_BAD_INO, &inode);
	if (inode.i_mode || inode.i_links_count)
	{
		printf ("Note: file system does not have a badblock inode.\n");
		if (test_disk)
			die ("You can't use -c or -t on this file system.\n", EXIT_USAGE);
		no_bad_inode = 1;
		return;
	}
	if (!inode_in_use (EXT2_BAD_INO))
	{
		printf ("The badblock inode is marked as unused. ");
		if (test_disk && !repair)
			die ("\nYou need to specify -a or -r to repair this.", EXIT_USAGE);
		if (ask ("Mark in use", 1))
			mark_inode (EXT2_BAD_INO);
	}

	name_depth = -1;    /* Flag to indicate we're parsing the bad blocks */
	for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
		add_block (dev, inode.i_block + i, 1);
	add_block_ind (dev, inode.i_block + EXT2_IND_BLOCK, 1);
	add_block_dind (dev, inode.i_block + EXT2_DIND_BLOCK, 1);
	add_block_tind (dev, inode.i_block + EXT2_TIND_BLOCK, 1);
	name_depth = 0;
}

void read_bad_blocks_from_file (const char * filename)
{
	FILE * f;
	unsigned long blockno;

	f = fopen (filename, "r");
	if (f == (FILE *) NULL)
		die ("Can't open file of bad blocks", EXIT_ERROR);
	while (!feof (f))
	{
		fscanf (f, "%d", &blockno);
		mark_bad (blockno);
		if (!block_in_use (blockno))
		{
			mark_block (blockno);
			FREEBLOCKSCOUNT--;
		}
	}
	fclose (f);
}

void display_bad_blocks (void)
{
	unsigned long block;
	int badblocks_count = 0;

	printf ("Bad blocks list:");
	for (block = 0; block < BLOCKS; block++)
		if (block_is_bad (block))
		{
			badblocks_count++;
			printf (" %d", block);
		}
	if (badblocks_count == 0)
		printf (" none\n");
	else
		printf ("\n");
}
